import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, Clock, CheckCircle, BookOpen, Calculator, Languages, FileText } from 'lucide-react'
import { motion } from 'framer-motion'
import './BookLesson.css'

const BookLesson = () => {
  const navigate = useNavigate()
  const [selectedSubject, setSelectedSubject] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [selectedTime, setSelectedTime] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const subjects = [
    { 
      id: 'math', 
      name: 'Математика', 
      icon: <Calculator size={28} strokeWidth={2} />, 
      description: 'Алгебра, геометрия, подготовка к ЕГЭ' 
    },
    { 
      id: 'physics', 
      name: 'Физика', 
      icon: <BookOpen size={28} strokeWidth={2} />, 
      description: 'Механика, оптика, термодинамика' 
    },
    { 
      id: 'english', 
      name: 'Английский', 
      icon: <Languages size={28} strokeWidth={2} />, 
      description: 'Грамматика, разговорная речь, подготовка к экзаменам' 
    },
    { 
      id: 'russian', 
      name: 'Русский', 
      icon: <FileText size={28} strokeWidth={2} />, 
      description: 'Орфография, пунктуация, сочинения' 
    }
  ]

  const timeSlots = [
    '09:00', '10:00', '11:00', '12:00', '13:00', '14:00',
    '15:00', '16:00', '17:00', '18:00', '19:00', '20:00'
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedSubject && selectedDate && selectedTime) {
      const savedLessons = localStorage.getItem('tutoringLessons')
      const lessons = savedLessons ? JSON.parse(savedLessons) : { active: [], past: [] }
      
      const subjectName = subjects.find(s => s.id === selectedSubject).name
      const tutors = {
        'math': 'Анна Петрова',
        'physics': 'Иван Сидоров',
        'english': 'Мария Козлова',
        'russian': 'Елена Смирнова'
      }
      
      const newLesson = {
        id: Date.now(),
        subject: subjectName,
        tutor: tutors[selectedSubject],
        date: selectedDate,
        time: selectedTime,
        status: 'active'
      }

      lessons.active.push(newLesson)
      localStorage.setItem('tutoringLessons', JSON.stringify(lessons))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedSubject('')
        setSelectedDate('')
        setSelectedTime('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="book-lesson">
      <section className="page-intro">
        <motion.div
          className="intro-content"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="intro-icon-wrapper">
            <Calendar className="intro-main-icon" size={56} color="#3B82F6" />
          </div>
          <h1 className="intro-title">Назначить урок</h1>
          <p className="intro-description">
            Выберите предмет, удобную дату и время для онлайн-занятия с репетитором
          </p>
        </motion.div>
      </section>

      <div className="container">
        <motion.section 
          className="booking-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <motion.form 
            className="booking-form" 
            onSubmit={handleSubmit}
            initial={{ scale: 0.98 }}
            animate={{ scale: 1 }}
            transition={{ type: "spring", stiffness: 100, damping: 10, delay: 0.3 }}
          >
            <div className="form-group">
              <label className="form-label">
                <BookOpen className="label-icon" size={18} strokeWidth={2} />
                Выберите предмет
              </label>
              <div className="subjects-grid">
                {subjects.map(subject => (
                  <motion.div
                    key={subject.id}
                    className={`subject-card ${selectedSubject === subject.id ? 'selected' : ''}`}
                    onClick={() => setSelectedSubject(subject.id)}
                    whileHover={{ y: -4, borderColor: 'rgba(59, 130, 246, 0.4)' }}
                    whileTap={{ scale: 0.98 }}
                    style={{ borderColor: selectedSubject === subject.id ? 'rgba(59, 130, 246, 0.5)' : 'var(--border)' }}
                  >
                    <div className="subject-icon-wrapper" style={{ borderColor: selectedSubject === subject.id ? 'rgba(59, 130, 246, 0.4)' : 'var(--border)' }}>
                      {subject.icon}
                    </div>
                    <div className="subject-content">
                      <h3 className="subject-name">{subject.name}</h3>
                      <p className="subject-description">{subject.description}</p>
                    </div>
                  </motion.div>
                ))}
              </div>
            </div>

            <div className="form-group">
              <label htmlFor="date" className="form-label">
                <Calendar className="label-icon" size={18} strokeWidth={2} />
                Дата урока
              </label>
              <input
                type="date"
                id="date"
                className="form-input"
                value={selectedDate}
                onChange={(e) => setSelectedDate(e.target.value)}
                min={minDate}
                max={maxDate}
                required
              />
            </div>

            <div className="form-group">
              <label htmlFor="time" className="form-label">
                <Clock className="label-icon" size={18} strokeWidth={2} />
                Время урока
              </label>
              <select
                id="time"
                className="form-select"
                value={selectedTime}
                onChange={(e) => setSelectedTime(e.target.value)}
                required
              >
                <option value="">Выберите время</option>
                {timeSlots.map(time => (
                  <option key={time} value={time}>{time}</option>
                ))}
              </select>
            </div>

            <motion.button 
              type="submit" 
              className="submit-btn" 
              disabled={!selectedSubject || !selectedDate || !selectedTime}
              whileHover={{ y: -2 }}
              whileTap={{ scale: 0.98 }}
            >
              <CheckCircle size={18} strokeWidth={2} className="btn-icon" />
              Назначить урок
            </motion.button>

            {isSubmitted && (
              <motion.div 
                className="success-message"
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.3 }}
              >
                <CheckCircle size={20} strokeWidth={2} className="success-icon" />
                <span>Урок успешно назначен!</span>
              </motion.div>
            )}
          </motion.form>
        </motion.section>

        <motion.section 
          className="info-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <div className="info-card">
            <h2 className="info-title">Преимущества онлайн-репетиторства</h2>
            <ul className="info-list">
              <li>Индивидуальный подход к каждому ученику</li>
              <li>Гибкий график занятий</li>
              <li>Опытные преподаватели с высшим образованием</li>
              <li>Интерактивные материалы и современные технологии</li>
              <li>Отслеживание прогресса и регулярная обратная связь</li>
            </ul>
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default BookLesson

